<?php
/**
 * PAINEL ADMIN - REFORÇADO & ULTRA RÁPIDO
 * Proteção contra Memory Leak e Estouro de RAM
 */

// Configurações de limite de memória (Blindagem preventiva)
ini_set('memory_limit', '128M'); 

// Configurações de Cookie antes de iniciar a sessão
ini_set('session.cookie_httponly', 1);
ini_set('session.use_only_cookies', 1);
ini_set('session.cookie_samesite', 'Lax');

session_start();

// BLINDAGEM: Redirecionamento precoce
if (isset($_SESSION["username"])) {
    header('Location: dashboard');
    exit;
}

/**
 * Carrega configurações de forma econômica
 */
function loadConfig() {
    static $config = null;
    if ($config === null) {
        $filePath = 'config.json';
        if (!file_exists($filePath)) {
            return ['usuario' => '', 'senha' => '', 'logo' => ''];
        }
        
        // Uso de streams para leitura se o arquivo for grande no futuro
        $jsonData = file_get_contents($filePath);
        $config = json_decode($jsonData, true);
        
        // Limpeza imediata da string bruta da memória
        unset($jsonData); 
        
        // Força a coleta de lixo se houver referências circulares
        if (function_exists('gc_collect_cycles')) {
            gc_collect_cycles();
        }
    }
    return $config;
}

$config = loadConfig();

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST["username"], $_POST["password"])) {
    // Sanitização sem criar duplicatas desnecessárias na memória
    $usernameInput = trim($_POST["username"]);
    
    // Comparação direta para evitar cópias de strings de senha na RAM
    if ($usernameInput === ($config['usuario'] ?? null) && $_POST["password"] === ($config['senha'] ?? null)) {
        $_SESSION["username"] = $usernameInput;
        session_regenerate_id(true);
        
        // Limpeza de variáveis pesadas antes do redirecionamento
        unset($config, $usernameInput);
        
        header('Location: dashboard'); 
        exit;
    } else {
        $error_message = 'Usuário ou senha inválidos!';
    }
}

// Otimização de saída: Limpa o buffer antes de enviar o HTML
if (ob_get_level()) ob_end_clean();
?>
<!DOCTYPE html>
<html lang="pt-br">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1, maximum-scale=1">
    <title>ACESSO ADMIN</title>
    
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;600;700&family=Orbitron:wght@700&display=swap" rel="stylesheet">

    <style>
        :root {
            --primary: #00ffe7;
            --primary-hover: #00d1bc;
            --background: #0a0a15;
            --glass-bg: rgba(20, 20, 30, 0.9);
            --pearl: #FBFCF8;
            --radius: 24px;
            --input-bg: rgba(255, 255, 255, 0.05);
        }

        * { 
            margin: 0; padding: 0; box-sizing: border-box;
            -webkit-tap-highlight-color: transparent;
        }

        body {
            min-height: 100vh;
            background: var(--background);
            font-family: 'Inter', sans-serif;
            display: flex;
            justify-content: center;
            align-items: center;
            overflow-x: hidden;
            position: relative;
            padding: 20px;
        }

        #particles {
            position: fixed;
            inset: 0;
            z-index: 0;
            pointer-events: none;
        }

        .login-container {
            position: relative;
            z-index: 10;
            background: var(--glass-bg);
            backdrop-filter: blur(25px);
            -webkit-backdrop-filter: blur(25px);
            border-radius: var(--radius);
            padding: 2.5rem;
            width: 100%;
            max-width: 420px;
            text-align: center;
            border: 1px solid rgba(255,255,255,0.08);
            box-shadow: 0 40px 100px -20px rgba(0, 0, 0, 0.8);
            animation: fadeIn 0.8s ease-out;
        }

        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(20px); }
            to { opacity: 1; transform: translateY(0); }
        }

        .logo {
            width: 90px;
            height: 90px;
            object-fit: contain;
            margin-bottom: 1.5rem;
            filter: drop-shadow(0 0 15px rgba(0, 255, 231, 0.4));
            transition: transform 0.3s ease;
        }
        
        .logo:hover { transform: scale(1.05); }

        .login-title {
            color: var(--pearl);
            font-family: 'Orbitron', sans-serif;
            font-size: 1.5rem;
            font-weight: 700;
            letter-spacing: 1px;
            margin-bottom: 2rem;
        }

        .input-group { 
            margin-bottom: 1.2rem; 
            position: relative;
        }

        input {
            width: 100%;
            padding: 1.1rem;
            background: var(--input-bg);
            border: 1px solid rgba(255, 255, 255, 0.1);
            border-radius: 16px;
            color: #fff;
            font-size: 1rem;
            text-align: center;
            transition: all 0.3s ease;
            user-select: text !important;
        }

        input:focus {
            border-color: var(--primary);
            background: rgba(255, 255, 255, 0.08);
            box-shadow: 0 0 20px rgba(0, 255, 231, 0.15);
            outline: none;
        }

        .error {
            background: rgba(255, 71, 71, 0.12);
            color: #ff6b6b;
            padding: 1rem;
            border-radius: 14px;
            margin-bottom: 1.5rem;
            font-size: 0.9rem;
            border: 1px solid rgba(255, 71, 71, 0.2);
            animation: shake 0.4s ease-in-out;
        }

        @keyframes shake {
            0%, 100% { transform: translateX(0); }
            25% { transform: translateX(-5px); }
            75% { transform: translateX(5px); }
        }

        button {
            width: 100%;
            padding: 1.1rem;
            background: var(--primary);
            color: #050510;
            border: none;
            border-radius: 16px;
            font-weight: 700;
            font-size: 1rem;
            cursor: pointer;
            text-transform: uppercase;
            letter-spacing: 1px;
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            margin-top: 0.5rem;
        }

        button:hover {
            background: var(--primary-hover);
            transform: translateY(-2px);
            box-shadow: 0 10px 20px rgba(0, 255, 231, 0.3);
        }

        /* Bloqueio de seleção apenas nos textos decorativos */
        h1, img { user-select: none; }
    </style>
</head>
<body oncontextmenu="return false;">

    <canvas id="particles"></canvas>

    <main class="login-container">
        <?php if (!empty($config['logo'])): ?>
            <img src="<?php echo htmlspecialchars($config['logo']); ?>" alt="Logo" class="logo">
        <?php endif; ?>
        
        <h1 class="login-title">ACESSO ADMIN</h1>

        <?php if (isset($error_message)): ?>
            <div class="error"><?php echo htmlspecialchars($error_message); ?></div>
        <?php endif; ?>

        <form method="post" action="">
            <div class="input-group">
                <input type="text" name="username" placeholder="Seu usuário" autocomplete="username" required>
            </div>
            <div class="input-group">
                <input type="password" name="password" placeholder="Sua senha" autocomplete="current-password" required>
            </div>
            <button type="submit">CONECTAR</button>
        </form>
    </main>

    <script>
        const canvas = document.getElementById('particles');
        const ctx = canvas.getContext('2d');
        let w, h, particles = [];

        function setCanvasSize() {
            w = canvas.width = window.innerWidth;
            h = canvas.height = window.innerHeight;
        }
        
        window.addEventListener('resize', () => {
            setCanvasSize();
            particles = [];
            initParticles();
        });
        setCanvasSize();

        class Particle {
            constructor() { this.reset(); }
            reset() {
                this.x = Math.random() * w;
                this.y = Math.random() * h;
                this.size = Math.random() * 2 + 0.5;
                this.speed = Math.random() * 0.4 + 0.1;
                this.opacity = Math.random() * 0.5 + 0.1;
            }
            update() { 
                this.y -= this.speed; 
                if (this.y < 0) this.reset(); 
            }
            draw() {
                ctx.fillStyle = `rgba(0, 255, 231, ${this.opacity})`;
                ctx.beginPath();
                ctx.arc(this.x, this.y, this.size, 0, Math.PI * 2);
                ctx.fill();
            }
        }

        function initParticles() {
            const count = window.innerWidth < 768 ? 30 : 70;
            for (let i = 0; i < count; i++) particles.push(new Particle());
        }

        function animate() {
            ctx.clearRect(0, 0, w, h);
            particles.forEach(p => { p.update(); p.draw(); });
            requestAnimationFrame(animate);
        }

        initParticles();
        animate();
    </script>
</body>
</html>